// AcMapOAddCmdRestoreBlocks.cpp: implementation of the AcMapOAddCmdRestoreBlocks class.
//
// (C) Copyright 2001 by Autodesk, Inc. 
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted, 
// provided that the above copyright notice appears in all copies and 
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting 
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC. 
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to 
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.

// CREATED BY:
// David Barbour, March 2001
//
// DESCRIPTION:
// 	Implementation of restore blocks functionality
//  This function gets AutoCAD blocks from Oracle.
//  Blocks cannot be create on the fly from an import reactor as the
//  AcDbBlockTable for the current drawing is not available for write 
//  during the ORAIMPORT function.
//  


#include "stdafx.h"
#include "stdarx.h"
#include "resource.h"

#include "OracleExports.h"	//for ORCL_API
#include "AdMapOracleConnection.h" // for AcMapOSEConnection
#include "AcMapOAddCmdRestoreBlocks.h"	// for class AcMapOReactorRestoreBlocks
#include "dbsymutl.h"	// for AcDbSymbolUtilities
#include "dbents.h"		// for AcDbPoint
#include "dbcfilrs.h"	// for AcDbWblockCloneFiler
#include "oracl.h"
#include <string>
#include <iostream.h>
#include <fstream.h>
#include <tchar.h>

// prototype for local static function
void
CreateBlock(ODynaset &Dynaset, AcDbDatabase *pDb, ODatabase& oDb);


//*****************************************************************************             
void AcMapOAddCmdRestoreBlock()
{
	AcMapOSEConnection * pConnect = AcMapOSEGetConnection();
	if(!pConnect->IsConnected())
	{
		acutPrintf("Not connected to Oracle.  Please connect before issuing this command\n");
		return;
	}

	acutPrintf("RESTORING BLOCKS\n");
	// Open a dynaset for the saved blocks.
	ODatabase oDb = pConnect->Database();
	AcDbDatabase *pDb = acdbHostApplicationServices()->workingDatabase();
	std::string strSql("Select * from ");
	strSql += pConnect->Schema();
	strSql += ".SAMPLE_BLOCKS";
	ODynaset ds;
	if(OSUCCESS == ds.Open(
		oDb, 
		strSql.c_str()))
	{
		for(ds.MoveFirst(); !ds.IsEOF(); ds.MoveNext())
		{
			CreateBlock(ds, pDb, oDb);
		}
	}
}



// This function restores a block to a drawing from an Oracle 
// dynaset which contains the block name and a BLOB containing
// the block definition.  Block attribute defs are also restored.
//
void
CreateBlock(ODynaset &dynaset, AcDbDatabase *pDb, ODatabase& oDb)
{
	// buffer to return the field contents
	char blockName[BLOCKNAME_MAX_LEN + 1];
	// number of bytes actually returned.
	unsigned short nNameLen = 0;
	oresult ores = dynaset.GetFieldValue(
		0,
		blockName,
		BLOCKNAME_MAX_LEN,
		&nNameLen);
	if(OSUCCESS != ores)
	{
		// Unknown error.
		return;
	}
	if(0 == nNameLen)
	{
		// this entity is not a block reference.
		return;
	}
	assert(pDb);
	if(!pDb)
	{
		return;
	}
	// does this block exist in the current database?
	AcDbObjectId blockId;
	Acad::ErrorStatus es = 
		AcDbSymbolUtilities::getBlockId (blockId, blockName, pDb);
	if(Acad::eOk == es)
	{
		// A block of this name exists in the database, import should proceed
		return;
	}

	// get the data representing the saved block
	OBlob oBlob;
	ores = dynaset.GetFieldValue(1, &oBlob);

	// no block exists, create one.
	
	// make a new file in the windows temp directory	
	char tempPath[_MAX_PATH];
	char fullPath[_MAX_PATH];
	char drive[_MAX_PATH];
	char dir[_MAX_PATH];

	::GetTempPath(_MAX_PATH, tempPath);
	_tsplitpath(tempPath, drive, dir, NULL, NULL);
	_tmakepath(fullPath, drive, dir, blockName, "dwg");

	fstream fs;
	fs.open(fullPath, ios::out);
	fs.setmode(filebuf::binary);
	
	unsigned long nBytesRead;
	unsigned char chunkp[CHUNK_LEN];
	unsigned short len = CHUNK_LEN;
	const char *pText;
	short status= OLOB_NEED_DATA;
	if(OSUCCESS == ores)
	{
		try
		{
			short status= OLOB_NEED_DATA;
			while(status == OLOB_NEED_DATA)
			{
				nBytesRead = oBlob.Read(&status, chunkp, len);
				fs.write(chunkp, nBytesRead);
			}
		}
		catch(OException &oExc)
		{
			pText = oExc.GetErrorText();
			pText = oExc.GetFailedMethodName();
			long errNum = oExc.GetErrorNumber();
		}
	}

	fs.close();

	assert(nBytesRead > 0);
	if(0 == nBytesRead)
	{
		// do some default behavior
		// create a block consisting of a single point
		// at origin (0,0,0)
		AcDbPoint *pPoint = new AcDbPoint;
		assert(pPoint);
		AcDbObjectId objId;
		AcDbBlockTable *pDbTable = NULL;
		es = pDb->getBlockTable(pDbTable, AcDb::kForWrite);
		if (es != Acad::eOk )
		{
			// can't get the block table, must fail here
			delete pPoint;
			return;
		}
		assert(pDbTable);
		AcDbBlockTableRecord *pBlock = new AcDbBlockTableRecord;
		es = pBlock->appendAcDbEntity(objId, pPoint);
		es = pDbTable->add(pBlock);
		pBlock->close();
		pDbTable->close();
	}
	else
	{
		// read in the block from a file.
		AcDbDatabase *pBlockDb = new AcDbDatabase;
		es = pBlockDb->readDwgFile(fullPath);
		if(Acad::eOk != es)
		{
			delete pBlockDb;
			return;
		}
		AcDbObjectId blockId;
		// insert the block into the working database
		es = pDb->insert(blockId, blockName, pBlockDb, false);
		delete pBlockDb;
	}
	::DeleteFile(fullPath);
}	




// eof